<?php
/**
 * Pinterest Pin Manager - MySQL Setup Wizard
 * This script initializes the MySQL database and configures the system for deployment.
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

$config_file = __DIR__ . '/config/db.php';
$schema_file = __DIR__ . '/schema.sql';
$lock_file = __DIR__ . '/setup.lock';

// 1. Check if already installed
if (file_exists($lock_file)) {
    die("Installation is locked. If you need to re-run setup, delete 'setup.lock' first.");
}

$step = $_GET['step'] ?? 1;
$error = '';
$success = '';

// Helper to check requirements
function check_requirements() {
    $results = [];
    $results['php_version'] = [
        'label' => 'PHP Version (>= 7.4)',
        'pass' => version_compare(PHP_VERSION, '7.4.0', '>='),
        'value' => PHP_VERSION
    ];
    $results['pdo_mysql'] = [
        'label' => 'PDO MySQL Extension',
        'pass' => extension_loaded('pdo_mysql'),
        'value' => extension_loaded('pdo_mysql') ? 'Loaded' : 'Missing'
    ];
    $results['config_write'] = [
        'label' => 'Config Directory Writeable',
        'pass' => is_writable(__DIR__ . '/config'),
        'value' => ''
    ];
    return $results;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'install') {
        $host = $_POST['db_host'] ?? 'localhost';
        $dbname = $_POST['db_name'] ?? '';
        $user = $_POST['db_user'] ?? '';
        $pass = $_POST['db_pass'] ?? '';

        try {
            // 1. Test Connection
            $dsn = "mysql:host=$host;charset=utf8mb4";
            $pdo = new PDO($dsn, $user, $pass);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

            // 2. Create Database if not exists
            $pdo->exec("CREATE DATABASE IF NOT EXISTS `$dbname` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            $pdo->exec("USE `$dbname` ");

            // 3. Create Users Table (Essential for login, missing from schema.sql)
            $pdo->exec("CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                username VARCHAR(255) NOT NULL UNIQUE,
                password VARCHAR(255) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

            // 4. Import Schema
            if (file_exists($schema_file)) {
                $sql = file_get_contents($schema_file);
                
                // Remove comments
                $sql = preg_replace('/--.*?\n/', '', $sql);
                
                // Split by semicolon, but be careful with multi-line statements
                $queries = explode(';', $sql);
                
                foreach ($queries as $query) {
                    $query = trim($query);
                    if (empty($query)) continue;
                    
                    // Skip database creation/use lines
                    if (stripos($query, 'CREATE DATABASE') === 0 || stripos($query, 'USE ') === 0) continue;
                    
                    try {
                        $pdo->exec($query);
                    } catch (PDOException $e) {
                        // Log and continue if it's already created, otherwise throw
                        if (stripos($e->getMessage(), 'already exists') === false) {
                            throw $e;
                        }
                    }
                }
            } else {
                throw new Exception("Schema file (schema.sql) not found in root directory.");
            }

            // 4. Create default admin user if users table exists and is empty
            $checkUser = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
            if ($checkUser == 0) {
                $stmt = $pdo->prepare("INSERT INTO users (username, password) VALUES (?, ?)");
                $stmt->execute(['admin', password_hash('pass123', PASSWORD_DEFAULT)]);
            }

            // 5. Write config/db.php
            $config_content = "<?php
/**
 * Database Configuration
 * Generated by Setup Wizard
 */

define('DB_HOST', '$host');
define('DB_NAME', '$dbname');
define('DB_USER', '$user');
define('DB_PASS', '$pass');
define('DB_CHARSET', 'utf8mb4');

function getDBConnection() {
    static \$pdo = null;
    if (\$pdo === null) {
        try {
            \$dsn = \"mysql:host=\" . DB_HOST . \";dbname=\" . DB_NAME . \";charset=\" . DB_CHARSET;
            \$options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];
            \$pdo = new PDO(\$dsn, DB_USER, DB_PASS, \$options);
        } catch (PDOException \$e) {
            die(\"Database connection failed: \" . \$e->getMessage());
        }
    }
    return \$pdo;
}

date_default_timezone_set('UTC');
";
            file_put_contents($config_file, $config_content);

            // 6. Create lock file
            file_put_contents($lock_file, date('Y-m-d H:i:s'));
            
            $success = "Installation complete!";
            $step = 3;
        } catch (Exception $e) {
            $error = "Installation failed: " . $e->getMessage();
        }
    }
}

$reqs = check_requirements();
$can_proceed = !in_array(false, array_column($reqs, 'pass'));

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MySQL Setup Wizard - Pinterest Pin Manager</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        :root { --primary: #667eea; --secondary: #764ba2; --bg: #f4f7fa; }
        body { font-family: 'Segoe UI', sans-serif; background: var(--bg); display: flex; align-items: center; justify-content: center; min-height: 100vh; margin: 0; }
        .card { background: white; width: 100%; max-width: 550px; padding: 40px; border-radius: 20px; box-shadow: 0 15px 35px rgba(0,0,0,0.1); }
        .header { text-align: center; margin-bottom: 30px; }
        .header i { font-size: 3em; color: var(--primary); margin-bottom: 10px; }
        .form-group { margin-bottom: 20px; }
        label { display: block; margin-bottom: 8px; font-weight: 600; color: #4a5568; }
        input { width: 100%; padding: 12px; border: 1px solid #e2e8f0; border-radius: 8px; box-sizing: border-box; }
        .btn { background: linear-gradient(135deg, var(--primary), var(--secondary)); color: white; border: none; padding: 15px; border-radius: 10px; cursor: pointer; font-weight: bold; width: 100%; font-size: 1.1em; }
        .alert { padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: center; }
        .alert-error { background: #fff5f5; color: #c53030; border: 1px solid #fed7d7; }
        .alert-success { background: #f0fff4; color: #2f855a; border: 1px solid #c6f6d5; }
        .req-item { display: flex; justify-content: space-between; padding: 10px 0; border-bottom: 1px solid #eee; }
    </style>
</head>
<body>

<div class="card">
    <div class="header">
        <i class="fas fa-database"></i>
        <h1>MySQL Setup</h1>
        <p>Deployment Wizard for cPanel</p>
    </div>

    <?php if ($error): ?><div class="alert alert-error"><?php echo $error; ?></div><?php endif; ?>
    <?php if ($success): ?><div class="alert alert-success"><?php echo $success; ?></div><?php endif; ?>

    <?php if ($step == 1): ?>
        <h3>Step 1: Requirements</h3>
        <div style="margin-bottom: 30px;">
            <?php foreach ($reqs as $r): ?>
                <div class="req-item">
                    <span><?php echo $r['label']; ?></span>
                    <span style="color: <?php echo $r['pass'] ? '#48bb78' : '#e53e3e'; ?>; font-weight: bold;">
                        <?php echo $r['pass'] ? 'OK' : 'FAIL'; ?>
                    </span>
                </div>
            <?php endforeach; ?>
        </div>
        <button onclick="window.location.href='?step=2'" class="btn" <?php echo !$can_proceed ? 'disabled' : ''; ?>>Next: Database Credentials</button>

    <?php elseif ($step == 2): ?>
        <h3>Step 2: Database Configuration</h3>
        <form method="POST">
            <input type="hidden" name="action" value="install">
            <div class="form-group">
                <label>Database Host</label>
                <input type="text" name="db_host" value="localhost" required>
            </div>
            <div class="form-group">
                <label>Database Name</label>
                <input type="text" name="db_name" placeholder="e.g. u123_pms" required>
            </div>
            <div class="form-group">
                <label>Database User</label>
                <input type="text" name="db_user" required>
            </div>
            <div class="form-group">
                <label>Database Password</label>
                <input type="password" name="db_pass">
            </div>
            <button type="submit" class="btn">Test & Install</button>
        </form>

    <?php elseif ($step == 3): ?>
        <h3>Step 3: Success!</h3>
        <p>Database configured and schema imported successfully.</p>
        <div style="background: #f7fafc; padding: 20px; border-radius: 10px; margin-bottom: 20px;">
            <strong>Admin Credentials:</strong><br>
            User: admin<br>
            Pass: pass123
        </div>
        <a href="admin/login.php" class="btn" style="text-decoration: none; display: block; text-align: center;">Go to Login</a>
        <p style="color: #e53e3e; margin-top: 20px; font-weight: bold; text-align: center;">
            <i class="fas fa-trash"></i> Please delete <code>setup.php</code> now!
        </p>
    <?php endif; ?>
</div>

</body>
</html>
