<?php
/**
 * CSV Import Script
 * Imports pins from CSV file with auto-category creation
 * 
 * CSV Format: title,description,imagelink,postlink,futerimg,category
 */

require_once __DIR__ . '/../config/db.php';

function importCSV($filepath) {
    if (!file_exists($filepath)) {
        return [
            'success' => false,
            'error' => 'File not found'
        ];
    }
    
    $pdo = getDBConnection();
    $imported = 0;
    $errors = [];
    $line = 0;
    
    try {
        $handle = fopen($filepath, 'r');
        
        if ($handle === false) {
            return [
                'success' => false,
                'error' => 'Could not open file'
            ];
        }
        
        // Skip header row
        $header = fgetcsv($handle);
        
        // Begin transaction for better performance
        $pdo->beginTransaction();
        
        while (($data = fgetcsv($handle)) !== false) {
            $line++;
            
            // Validate row has enough columns
            if (count($data) < 6) {
                $errors[] = "Line $line: Insufficient columns";
                continue;
            }
            
            // Parse CSV data
            $title = trim($data[0]);
            $description = trim($data[1]);
            $imagelink = trim($data[2]);
            $postlink = trim($data[3]);
            $futerimg = isset($data[4]) ? (int)$data[4] : 0;
            $category = trim($data[5]);
            
            // Validate required fields
            if (empty($title) || empty($imagelink) || empty($category)) {
                $errors[] = "Line $line: Missing required fields (title, imagelink, or category)";
                continue;
            }
            
            // Normalize category name
            $category = ucwords(strtolower($category));
            
            try {
                // Insert or get category
                $categoryStmt = $pdo->prepare(
                    "INSERT INTO categories (category_name) VALUES (:category) 
                     ON DUPLICATE KEY UPDATE id=LAST_INSERT_ID(id)"
                );
                $categoryStmt->execute(['category' => $category]);
                $category_id = $pdo->lastInsertId();
                
                // Insert pin
                $pinStmt = $pdo->prepare(
                    "INSERT INTO pins (title, description, imagelink, postlink, futerimg, category_id, inserted_date) 
                     VALUES (:title, :description, :imagelink, :postlink, :futerimg, :category_id, NOW())"
                );
                
                $pinStmt->execute([
                    'title' => $title,
                    'description' => $description,
                    'imagelink' => $imagelink,
                    'postlink' => $postlink,
                    'futerimg' => $futerimg,
                    'category_id' => $category_id
                ]);
                
                $imported++;
                
            } catch (PDOException $e) {
                $errors[] = "Line $line: " . $e->getMessage();
            }
        }
        
        fclose($handle);
        
        // Commit transaction
        $pdo->commit();
        
        return [
            'success' => true,
            'imported' => $imported,
            'errors' => $errors,
            'total_lines' => $line
        ];
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'imported' => $imported,
            'errors' => $errors
        ];
    }
}

// CLI usage
if (php_sapi_name() === 'cli') {
    if ($argc < 2) {
        echo "Usage: php import_csv.php <csv_file_path>\n";
        exit(1);
    }
    
    $result = importCSV($argv[1]);
    
    if ($result['success']) {
        echo "Import completed successfully!\n";
        echo "Imported: {$result['imported']} pins\n";
        echo "Total lines processed: {$result['total_lines']}\n";
        
        if (!empty($result['errors'])) {
            echo "\nErrors:\n";
            foreach ($result['errors'] as $error) {
                echo "  - $error\n";
            }
        }
    } else {
        echo "Import failed: {$result['error']}\n";
        if (isset($result['errors']) && !empty($result['errors'])) {
            echo "\nErrors:\n";
            foreach ($result['errors'] as $error) {
                echo "  - $error\n";
            }
        }
    }
}
