<?php
/**
 * Get Pin API - GLOBAL BOARD UNIQUENESS VERSION
 * Returns a random unposted pin with board information
 * 
 * RULE: Each board template can only be used by ONE account EVER
 * Example: If account1 uses "Fashion Trends" (bid: 3), 
 *          account2 CANNOT use "Fashion Trends" - it's taken!
 * 
 * Usage: GET get_pin_api.php?acc_username=acc1&account_date=YYYY-MM-DD
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';

// Validate input
$acc_username = isset($_GET['acc_username']) ? trim($_GET['acc_username']) : '';
$account_date = isset($_GET['account_date']) ? trim($_GET['account_date']) : '';

if (empty($acc_username)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'acc_username is required'
    ]);
    exit;
}

if (empty($account_date)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'account_date is required (format: YYYY-MM-DD)'
    ]);
    exit;
}

// Validate date format
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $account_date)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Invalid date format. Use YYYY-MM-DD'
    ]);
    exit;
}

try {
    $pdo = getDBConnection();
    
    // Calculate account age in days
    $today = date('Y-m-d');
    $accountAge = (strtotime($today) - strtotime($account_date)) / (60 * 60 * 24);
    
    // Get appropriate stage based on account age
    $stageSql = "SELECT * FROM stages 
                 WHERE old_day <= :account_age 
                 ORDER BY old_day DESC 
                 LIMIT 1";
    
    $stageStmt = $pdo->prepare($stageSql);
    $stageStmt->execute(['account_age' => $accountAge]);
    $stage = $stageStmt->fetch();
    
    if (!$stage) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'No posting stage configured for account age: ' . (int)$accountAge . ' days'
        ]);
        exit;
    }
    
    // Check how many pins this account has posted TODAY
    $countSql = "SELECT COUNT(*) as posted_today 
                 FROM pins 
                 WHERE acc_username = :acc_username 
                 AND DATE(account_date) = CURDATE()";
    
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute(['acc_username' => $acc_username]);
    $postedToday = $countStmt->fetch()['posted_today'];
    
    // Check if daily limit reached
    if ($postedToday >= $stage['total_pins_perDay']) {
        http_response_code(200);
        echo json_encode([
            'success' => false,
            'error' => 'Daily posting limit reached',
            'details' => [
                'posted_today' => (int)$postedToday,
                'daily_limit' => (int)$stage['total_pins_perDay'],
                'account_age_days' => (int)$accountAge,
                'current_stage' => $stage['name']
            ]
        ]);
        exit;
    }
    
    // Select one random unposted pin
    $pinSql = "SELECT id, title, description, imagelink, postlink, futerimg, category_id 
               FROM pins 
               WHERE pin_id IS NULL 
               ORDER BY RAND() 
               LIMIT 1";
    
    $pinStmt = $pdo->query($pinSql);
    $pin = $pinStmt->fetch();
    
    if (!$pin) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'No available pins found in database'
        ]);
        exit;
    }
    
    // Check if THIS account already has a board for this category
    $boardCheckSql = "SELECT ab.bord_id, ab.bid
                      FROM account_board ab
                      INNER JOIN board_data bd ON ab.bid = bd.bid
                      WHERE ab.acc_username = :acc_username 
                      AND bd.category_id = :category_id
                      LIMIT 1";
    
    $boardCheckStmt = $pdo->prepare($boardCheckSql);
    $boardCheckStmt->execute([
        'acc_username' => $acc_username,
        'category_id' => $pin['category_id']
    ]);
    $existingBoard = $boardCheckStmt->fetch();
    
    if ($existingBoard) {
        // This account already has a board for this category - reuse it
        $response = [
            'success' => true,
            'pin' => [
                'id' => (int)$pin['id'],
                'title' => $pin['title'],
                'description' => $pin['description'],
                'imagelink' => $pin['imagelink'],
                'postlink' => $pin['postlink'],
                'futerimg' => (int)$pin['futerimg'],
                'category_id' => (int)$pin['category_id']
            ],
            'board' => [
                'create' => false,
                'bord_id' => $existingBoard['bord_id'],
                'bid' => (int)$existingBoard['bid']
            ],
            'stage_info' => [
                'account_age_days' => (int)$accountAge,
                'current_stage' => $stage['name'],
                'posted_today' => (int)$postedToday,
                'daily_limit' => (int)$stage['total_pins_perDay'],
                'remaining_today' => (int)($stage['total_pins_perDay'] - $postedToday)
            ]
        ];
    } else {
        // GLOBAL UNIQUENESS: Select board template that has NEVER been used by ANY account
        $templateSql = "SELECT bid, name, description 
                        FROM board_data 
                        WHERE category_id = :category_id 
                        AND bid NOT IN (
                            SELECT DISTINCT bid FROM account_board
                        )
                        ORDER BY RAND() 
                        LIMIT 1";
        
        $templateStmt = $pdo->prepare($templateSql);
        $templateStmt->execute(['category_id' => $pin['category_id']]);
        $template = $templateStmt->fetch();
        
        if (!$template) {
            // All board templates for this category are taken by other accounts
            http_response_code(200);
            echo json_encode([
                'success' => false,
                'error' => 'No available board templates for this category',
                'details' => [
                    'category_id' => (int)$pin['category_id'],
                    'reason' => 'All board templates for this category are already used by other accounts',
                    'suggestion' => 'Add more board templates for this category in the dashboard'
                ]
            ]);
            exit;
        }
        
        $response = [
            'success' => true,
            'pin' => [
                'id' => (int)$pin['id'],
                'title' => $pin['title'],
                'description' => $pin['description'],
                'imagelink' => $pin['imagelink'],
                'postlink' => $pin['postlink'],
                'futerimg' => (int)$pin['futerimg'],
                'category_id' => (int)$pin['category_id']
            ],
            'board' => [
                'create' => true,
                'bid' => (int)$template['bid'],
                'name' => $template['name'],
                'description' => $template['description'],
                'bord_id' => null
            ],
            'stage_info' => [
                'account_age_days' => (int)$accountAge,
                'current_stage' => $stage['name'],
                'posted_today' => (int)$postedToday,
                'daily_limit' => (int)$stage['total_pins_perDay'],
                'remaining_today' => (int)($stage['total_pins_perDay'] - $postedToday)
            ],
            'board_usage_info' => [
                'message' => 'This board template is now reserved for this account',
                'note' => 'No other account can use this board template'
            ]
        ];
    }
    
    http_response_code(200);
    echo json_encode($response, JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    error_log("Get Pin API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Internal server error'
    ]);
}