<?php
/**
 * Pin Update API - FINAL VERSION WITH PROPER STATUS CODES
 * Updates pin information after it has been posted to Pinterest
 * 
 * Usage: GET api_edit.php?id=123&pin_id=PIN123&acc_username=acc1&bord_id=99887766
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';

// Validate input
$id = isset($_GET['id']) ? trim($_GET['id']) : '';
$pin_id = isset($_GET['pin_id']) ? trim($_GET['pin_id']) : '';
$acc_username = isset($_GET['acc_username']) ? trim($_GET['acc_username']) : '';
$bord_id = isset($_GET['bord_id']) ? trim($_GET['bord_id']) : '';

if (empty($id) || empty($pin_id) || empty($acc_username) || empty($bord_id)) {
    http_response_code(400); // Bad Request
    echo json_encode([
        'success' => false,
        'error' => 'id, pin_id, acc_username, and bord_id are required'
    ]);
    exit;
}

// Validate id is numeric
if (!is_numeric($id)) {
    http_response_code(400); // Bad Request
    echo json_encode([
        'success' => false,
        'error' => 'id must be a valid number'
    ]);
    exit;
}

try {
    $pdo = getDBConnection();
    
    // Update pin only if it hasn't been posted yet (pin_id IS NULL)
    // This prevents double-posting and ensures data integrity
    $sql = "UPDATE pins SET
            pin_id = :pin_id,
            acc_username = :acc_username,
            account_date = CURDATE(),
            bord_id = :bord_id
            WHERE id = :id
            AND pin_id IS NULL";
    
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute([
        'pin_id' => $pin_id,
        'acc_username' => $acc_username,
        'bord_id' => $bord_id,
        'id' => $id
    ]);
    
    $rowsAffected = $stmt->rowCount();
    
    if ($rowsAffected > 0) {
        http_response_code(200); // OK
        echo json_encode([
            'success' => true,
            'message' => 'Pin updated successfully',
            'data' => [
                'id' => (int)$id,
                'pin_id' => $pin_id,
                'acc_username' => $acc_username,
                'bord_id' => $bord_id,
                'account_date' => date('Y-m-d')
            ]
        ], JSON_PRETTY_PRINT);
    } else {
        // Pin was already posted or doesn't exist
        $checkStmt = $pdo->prepare("SELECT pin_id FROM pins WHERE id = :id");
        $checkStmt->execute(['id' => $id]);
        $pin = $checkStmt->fetch();
        
        if (!$pin) {
            http_response_code(404); // Not Found
            echo json_encode([
                'success' => false,
                'error' => 'Pin not found'
            ]);
        } else {
            http_response_code(409); // Conflict
            echo json_encode([
                'success' => false,
                'error' => 'Pin has already been posted',
                'existing_pin_id' => $pin['pin_id']
            ]);
        }
    }
    
} catch (PDOException $e) {
    error_log("Pin Update Error: " . $e->getMessage());
    http_response_code(500); // Internal Server Error
    echo json_encode([
        'success' => false,
        'error' => 'Database error occurred'
    ]);
} catch (Exception $e) {
    error_log("Pin Update Error: " . $e->getMessage());
    http_response_code(500); // Internal Server Error
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}