<?php
/**
 * Board Registration API - FINAL VERSION WITH PROPER STATUS CODES
 * Registers a newly created Pinterest board for an account
 * 
 * Usage: GET api_databord.php?acc_username=acc1&bid=5&bord_id=99887766
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';

// Validate input
$acc_username = isset($_GET['acc_username']) ? trim($_GET['acc_username']) : '';
$bid = isset($_GET['bid']) ? trim($_GET['bid']) : '';
$bord_id = isset($_GET['bord_id']) ? trim($_GET['bord_id']) : '';

if (empty($acc_username) || empty($bid) || empty($bord_id)) {
    http_response_code(400); // Bad Request
    echo json_encode([
        'success' => false,
        'error' => 'acc_username, bid, and bord_id are required'
    ]);
    exit;
}

// Validate bid is numeric
if (!is_numeric($bid)) {
    http_response_code(400); // Bad Request
    echo json_encode([
        'success' => false,
        'error' => 'bid must be a valid number'
    ]);
    exit;
}

try {
    $pdo = getDBConnection();
    
    // Check if board template exists
    $checkStmt = $pdo->prepare("SELECT bid FROM board_data WHERE bid = :bid");
    $checkStmt->execute(['bid' => $bid]);
    
    if (!$checkStmt->fetch()) {
        http_response_code(404); // Not Found
        echo json_encode([
            'success' => false,
            'error' => 'Board template not found',
            'bid_requested' => (int)$bid
        ]);
        exit;
    }
    
    // Check if this account+bid combination already exists
    $existingStmt = $pdo->prepare("SELECT id, bord_id FROM account_board WHERE acc_username = :acc_username AND bid = :bid");
    $existingStmt->execute([
        'acc_username' => $acc_username,
        'bid' => $bid
    ]);
    $existing = $existingStmt->fetch();
    
    if ($existing) {
        // Update existing record
        $updateSql = "UPDATE account_board 
                      SET bord_id = :bord_id, creation_date = NOW() 
                      WHERE acc_username = :acc_username AND bid = :bid";
        
        $updateStmt = $pdo->prepare($updateSql);
        $result = $updateStmt->execute([
            'bord_id' => $bord_id,
            'acc_username' => $acc_username,
            'bid' => $bid
        ]);
        
        if ($result) {
            http_response_code(200); // OK
            echo json_encode([
                'success' => true,
                'message' => 'Board updated successfully (already existed)',
                'action' => 'updated',
                'data' => [
                    'acc_username' => $acc_username,
                    'bord_id' => $bord_id,
                    'bid' => (int)$bid,
                    'previous_bord_id' => $existing['bord_id']
                ]
            ], JSON_PRETTY_PRINT);
        } else {
            throw new Exception('Failed to update board');
        }
    } else {
        // Insert new record
        $insertSql = "INSERT INTO account_board (acc_username, bord_id, bid, creation_date) 
                      VALUES (:acc_username, :bord_id, :bid, NOW())";
        
        $insertStmt = $pdo->prepare($insertSql);
        $result = $insertStmt->execute([
            'acc_username' => $acc_username,
            'bord_id' => $bord_id,
            'bid' => $bid
        ]);
        
        if ($result) {
            http_response_code(201); // Created
            echo json_encode([
                'success' => true,
                'message' => 'Board registered successfully',
                'action' => 'created',
                'data' => [
                    'acc_username' => $acc_username,
                    'bord_id' => $bord_id,
                    'bid' => (int)$bid,
                    'id' => (int)$pdo->lastInsertId()
                ]
            ], JSON_PRETTY_PRINT);
        } else {
            throw new Exception('Failed to register board');
        }
    }
    
} catch (PDOException $e) {
    error_log("Board Registration Error: " . $e->getMessage());
    http_response_code(500); // Internal Server Error
    echo json_encode([
        'success' => false,
        'error' => 'Database error occurred',
        'debug' => [
            'message' => $e->getMessage(),
            'code' => $e->getCode()
        ]
    ]);
} catch (Exception $e) {
    error_log("Board Registration Error: " . $e->getMessage());
    http_response_code(500); // Internal Server Error
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}