<?php
require_once __DIR__ . '/../config/db.php';

$message = '';
$messageType = '';

// Handle initial file upload for AJAX processing
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'upload_temp') {
    header('Content-Type: application/json');
    if ($_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
        $temp_dir = __DIR__ . '/../import';
        if (!is_dir($temp_dir)) mkdir($temp_dir, 0777, true);
        
        $filename = 'temp_' . basename($_FILES['csv_file']['name']);
        $target = $temp_dir . '/' . $filename;
        
        if (move_uploaded_file($_FILES['csv_file']['tmp_name'], $target)) {
            // Count total lines for progress bar
            $linecount = 0;
            $handle = fopen($target, "r");
            while(!feof($handle)){
              $line = fgets($handle);
              $linecount++;
            }
            fclose($handle);
            
            echo json_encode(['success' => true, 'filename' => $_FILES['csv_file']['name'], 'total' => $linecount - 1]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Failed to save temporary file']);
        }
    } else {
        echo json_encode(['success' => false, 'error' => 'Upload error: ' . $_FILES['csv_file']['error']]);
    }
    exit;
}

$page_title = 'Upload CSV - Pinterest Pin Manager';
require_once 'includes/header.php';
?>

<div class="header-content">
    <h1>📤 Upload CSV File</h1>
</div>

<div class="page-card">
    <div id="status-container" style="display: none; margin-bottom: 30px;">
        <div style="display: flex; justify-content: space-between; margin-bottom: 10px;">
            <strong id="progress-text">Processing: 0%</strong>
            <span id="progress-stats">0 / 0 pins</span>
        </div>
        <div style="width: 100%; background: #edf2f7; height: 12px; border-radius: 6px; overflow: hidden;">
            <div id="progress-bar" style="width: 0%; height: 100%; background: linear-gradient(90deg, var(--primary), var(--secondary)); transition: width 0.3s ease;"></div>
        </div>
        <p id="current-status" style="margin-top: 10px; color: #718096; font-size: 0.9em; text-align: center;">Preparing to start...</p>
    </div>

    <div id="import-result" class="alert" style="display: none;"></div>
    
    <div id="form-container">
        <form method="POST" enctype="multipart/form-data" id="upload-form">
            <div id="upload-area" style="border: 3px dashed var(--primary); border-radius: 10px; padding: 50px; text-align: center; cursor: pointer; transition: var(--transition); margin-bottom: 20px;">
                <div style="font-size: 4em; margin-bottom: 15px;">📁</div>
                <h3>Click to upload or drag and drop</h3>
                <p>CSV file with pins data</p>
            </div>
            
            <input type="file" name="csv_file" id="csv_file" accept=".csv" required style="display: none;">
            
            <div id="file-info" style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; display: none;">
                <strong>Selected file:</strong> <span id="file-name"></span>
            </div>
            
            <button type="submit" class="btn-submit" id="submit-btn" disabled style="width: 100%;">Upload and Import</button>
        </form>
    </div>
    
    <div style="background: #fff3cd; border: 1px solid #ffc107; padding: 20px; border-radius: 8px; margin-top: 20px;">
        <h3>📋 CSV Format Requirements</h3>
        <p><strong>Columns (in order):</strong></p>
        <code>title,description,imagelink,postlink,futerimg,category</code>
    </div>
</div>

<script>
    const fileInput = document.getElementById('csv_file');
    const uploadArea = document.getElementById('upload-area');
    const fileInfo = document.getElementById('file-info');
    const fileName = document.getElementById('file-name');
    const submitBtn = document.getElementById('submit-btn');
    const form = document.getElementById('upload-form');
    
    uploadArea.addEventListener('click', () => fileInput.click());
    
    fileInput.addEventListener('change', (e) => {
        if (e.target.files.length > 0) {
            fileName.textContent = e.target.files[0].name;
            fileInfo.style.display = 'block';
            submitBtn.disabled = false;
        }
    });

    form.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const formData = new FormData(form);
        formData.append('action', 'upload_temp');
        
        submitBtn.disabled = true;
        uploadArea.style.opacity = '0.5';
        
        try {
            // Step 1: Upload the file
            const response = await fetch('upload_csv.php', { method: 'POST', body: formData });
            const data = await response.json();
            
            if (data.success) {
                document.getElementById('form-container').style.display = 'none';
                document.getElementById('status-container').style.display = 'block';
                startImport(data.filename, data.total);
            } else {
                alert('Upload failed: ' + data.error);
                submitBtn.disabled = false;
                uploadArea.style.opacity = '1';
            }
        } catch (error) {
            console.error('Error:', error);
            alert('An error occurred during upload.');
            submitBtn.disabled = false;
        }
    });

    async function startImport(filename, total) {
        let offset = 0;
        let imported = 0;
        let done = false;
        
        while (!done) {
            const formData = new FormData();
            formData.append('type', 'pins');
            formData.append('filename', filename);
            formData.append('offset', offset);
            formData.append('limit', 50); // Process in batches of 50
            
            try {
                const response = await fetch('api_import_chunk.php', { method: 'POST', body: formData });
                const result = await response.json();
                
                if (result.success) {
                    offset = result.next_offset;
                    imported += result.imported;
                    done = result.done;
                    
                    // Update progress
                    const pct = Math.min(100, Math.round((offset / total) * 100));
                    document.getElementById('progress-bar').style.width = pct + '%';
                    document.getElementById('progress-text').textContent = `Processing: ${pct}%`;
                    document.getElementById('progress-stats').textContent = `${imported} / ${total} pins`;
                    document.getElementById('current-status').textContent = `Imported batch of ${result.imported} pins...`;
                    
                    if (done) {
                        finishImport(imported, total);
                    }
                } else {
                    showError(result.error);
                    break;
                }
            } catch (error) {
                showError('Network error occurred during processing.');
                break;
            }
        }
    }

    function finishImport(count, total) {
        document.getElementById('progress-bar').style.background = '#48bb78';
        document.getElementById('current-status').textContent = 'Import Finished!';
        
        const resultDiv = document.getElementById('import-result');
        resultDiv.className = 'alert alert-success';
        resultDiv.style.display = 'block';
        resultDiv.innerHTML = `<i class="fas fa-check-circle"></i> Completed! Successfully imported <strong>${count}</strong> pins.`;
        
        setTimeout(() => {
            window.location.href = 'pins_list.php';
        }, 2000);
    }

    function showError(msg) {
        document.getElementById('progress-bar').style.background = '#e53e3e';
        document.getElementById('current-status').textContent = 'Error occurred';
        document.getElementById('current-status').style.color = '#e53e3e';
        
        const resultDiv = document.getElementById('import-result');
        resultDiv.className = 'alert alert-error';
        resultDiv.style.display = 'block';
        resultDiv.innerHTML = `<i class="fas fa-exclamation-triangle"></i> <strong>Import Failed:</strong> ${msg}`;
    }

    uploadArea.addEventListener('dragover', (e) => { e.preventDefault(); uploadArea.style.background = '#f0f0ff'; });
    uploadArea.addEventListener('dragleave', () => { uploadArea.style.background = 'transparent'; });
    uploadArea.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadArea.style.background = 'transparent';
        if (e.dataTransfer.files.length > 0) {
            fileInput.files = e.dataTransfer.files;
            fileName.textContent = e.dataTransfer.files[0].name;
            fileInfo.style.display = 'block';
            submitBtn.disabled = false;
        }
    });
</script>

<?php require_once 'includes/footer.php'; ?>
