<?php
require_once __DIR__ . '/../config/db.php';
$pdo = getDBConnection();
$message = ''; $messageType = '';

// Handle AJAX temporary upload for boards
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'upload_temp_boards') {
    header('Content-Type: application/json');
    if ($_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
        $temp_dir = __DIR__ . '/../import';
        if (!is_dir($temp_dir)) mkdir($temp_dir, 0777, true);
        
        $filename = 'temp_boards_' . basename($_FILES['csv_file']['name']);
        $target = $temp_dir . '/' . $filename;
        
        if (move_uploaded_file($_FILES['csv_file']['tmp_name'], $target)) {
            $linecount = 0;
            $handle = fopen($target, "r");
            while(!feof($handle)){ fgets($handle); $linecount++; }
            fclose($handle);
            echo json_encode(['success' => true, 'filename' => $filename, 'total' => $linecount - 1]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Failed to save file']);
        }
    } else {
        echo json_encode(['success' => false, 'error' => 'Upload error']);
    }
    exit;
}

// Handle board template deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_board'])) {
    try {
        $pdo->prepare("DELETE FROM board_data WHERE bid = ?")->execute([(int)$_POST['board_bid']]);
        $message = "Board template deleted successfully!"; $messageType = "success";
    } catch (PDOException $e) { $message = "Error deleting board: " . $e->getMessage(); $messageType = "error"; }
}

// Handle single board addition
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_board'])) {
    $name = trim($_POST['board_name']);
    $description = trim($_POST['board_description']);
    $categoryId = (int)$_POST['category_id'];
    if (!empty($name) && $categoryId > 0) {
        try {
            $pdo->prepare("INSERT INTO board_data (name, description, category_id) VALUES (:name, :description, :category_id)")->execute(['name'=>$name, 'description'=>$description, 'category_id'=>$categoryId]);
            $message = "Board template added successfully!"; $messageType = "success";
        } catch (PDOException $e) { $message = "Error adding board: " . $e->getMessage(); $messageType = "error"; }
    }
}

$categories = $pdo->query("SELECT id, category_name FROM categories ORDER BY category_name")->fetchAll();
$boards = $pdo->query("SELECT bd.*, c.category_name, COUNT(DISTINCT ab.acc_username) as usage_count FROM board_data bd LEFT JOIN categories c ON bd.category_id = c.id LEFT JOIN account_board ab ON bd.bid = ab.bid GROUP BY bd.bid ORDER BY c.category_name, bd.name")->fetchAll();

$page_title = 'Board Templates - Pinterest Pin Manager';
require_once 'includes/header.php';
?>

<div class="header-content">
    <h1><i class="fas fa-columns"></i> Board Management</h1>
</div>

<div id="import-result-msg" class="alert" style="display: none; margin-bottom: 20px;"></div>

<?php if ($message): ?>
    <div class="alert alert-<?php echo $messageType; ?>"><?php echo $message; ?></div>
<?php endif; ?>

<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(400px, 1fr)); gap: 30px; margin-bottom: 30px;">
    <!-- Manual Add -->
    <div class="page-card" style="margin-bottom: 0;">
        <h2><i class="fas fa-plus-circle"></i> Add Single Template</h2>
        <form method="POST" style="margin-top: 15px;">
            <div class="form-group">
                <label>Board Name *</label>
                <input type="text" name="board_name" class="form-control" required placeholder="e.g. Kitchen Decor">
            </div>
            <div class="form-group">
                <label>Category *</label>
                <select name="category_id" class="form-control" required>
                    <option value="">Select Category</option>
                    <?php foreach ($categories as $cat): ?>
                        <option value="<?php echo $cat['id']; ?>"><?php echo htmlspecialchars($cat['category_name']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="form-group">
                <label>Description</label>
                <textarea name="board_description" class="form-control" style="min-height: 80px;" placeholder="Optional description..."></textarea>
            </div>
            <button type="submit" name="add_board" class="btn-submit" style="width: 100%;">Add Board Template</button>
        </form>
    </div>

    <!-- CSV Import -->
    <div class="page-card" style="margin-bottom: 0;">
        <div id="boards-import-form">
            <h2><i class="fas fa-file-upload"></i> Bulk Import CSV</h2>
            <p style="color: #718096; font-size: 0.9em; margin: 10px 0 20px;">Upload a CSV to add many boards. Format: Name, Description, Category.</p>
            
            <form id="board-csv-form">
                <div style="border: 2px dashed #e2e8f0; padding: 30px; border-radius: 10px; text-align: center; margin-bottom: 20px;" id="drop-zone-boards">
                    <input type="file" name="csv_file" accept=".csv" required id="csv_input_boards" style="display: none;">
                    <label for="csv_input_boards" style="cursor: pointer;">
                        <i class="fas fa-cloud-upload-alt" style="font-size: 2.5em; color: var(--primary); margin-bottom: 10px;"></i>
                        <div id="file_name_boards">Click to select CSV file</div>
                    </label>
                </div>
                <button type="submit" id="boards-submit-btn" class="btn-submit" style="width: 100%; background: linear-gradient(135deg, #4fd1c5 0%, #38b2ac 100%);" disabled>Start Bulk Import</button>
            </form>
        </div>

        <div id="boards-progress-container" style="display: none;">
            <h3><i class="fas fa-spinner fa-spin"></i> Importing Boards...</h3>
            <div style="display: flex; justify-content: space-between; margin: 15px 0 5px;">
                <span id="board-pct">0%</span>
                <span id="board-counts">0 / 0</span>
            </div>
            <div style="width: 100%; background: #edf2f7; height: 10px; border-radius: 5px; overflow: hidden;">
                <div id="board-progress-bar" style="width: 0%; height: 100%; background: #38b2ac; transition: width 0.3s;"></div>
            </div>
            <p id="board-status" style="font-size: 0.8em; color: #718096; margin-top: 10px;">Starting...</p>
        </div>
    </div>
</div>

<div class="page-card">
    <h2><i class="fas fa-list"></i> Existing Board Templates</h2>
    <div style="overflow-x: auto; margin-top: 20px;">
        <table>
            <thead>
                <tr>
                    <th>Board Name</th>
                    <th>Category</th>
                    <th style="text-align: center;">In Use By</th>
                    <th style="text-align: right;">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($boards)): ?>
                    <tr><td colspan="4" style="text-align: center; color: #999; padding: 40px;">No templates found.</td></tr>
                <?php else: ?>
                    <?php foreach ($boards as $board): ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($board['name']); ?></strong><br>
                                <small style="color: #718096;"><?php echo htmlspecialchars($board['description']); ?></small>
                            </td>
                            <td><span class="badge badge-info"><?php echo htmlspecialchars($board['category_name']); ?></span></td>
                            <td style="text-align: center;"><span class="badge <?php echo $board['usage_count'] > 0 ? 'badge-success' : 'badge-warning'; ?>"><?php echo $board['usage_count']; ?> accounts</span></td>
                            <td style="text-align: right;">
                                <form method="POST" onsubmit="return confirm('Delete this template?');" style="display: inline;">
                                    <input type="hidden" name="board_bid" value="<?php echo $board['bid']; ?>">
                                    <button type="submit" name="delete_board" style="background: none; border: none; color: #e53e3e; cursor: pointer;"><i class="fas fa-trash-alt"></i></button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<script>
    const boardInput = document.getElementById('csv_input_boards');
    const boardLabel = document.getElementById('file_name_boards');
    const boardSubmit = document.getElementById('boards-submit-btn');
    const boardForm = document.getElementById('board-csv-form');

    boardInput.onchange = function() {
        if (this.files[0]) {
            boardLabel.textContent = this.files[0].name;
            boardSubmit.disabled = false;
        }
    };

    boardForm.onsubmit = async function(e) {
        e.preventDefault();
        boardSubmit.disabled = true;
        const fd = new FormData(boardForm);
        fd.append('action', 'upload_temp_boards');

        try {
            const res = await fetch('boards.php', { method: 'POST', body: fd });
            const data = await res.json();
            if (data.success) {
                document.getElementById('boards-import-form').style.display = 'none';
                document.getElementById('boards-progress-container').style.display = 'block';
                processBoards(data.filename, data.total);
            } else {
                alert(data.error);
                boardSubmit.disabled = false;
            }
        } catch (err) {
            alert('Upload error');
            boardSubmit.disabled = false;
        }
    };

    async function processBoards(filename, total) {
        let offset = 0;
        let imported = 0;
        let done = false;

        while (!done) {
            const fd = new FormData();
            fd.append('type', 'boards');
            fd.append('filename', filename);
            fd.append('offset', offset);
            fd.append('limit', 50);

            try {
                const res = await fetch('api_import_chunk.php', { method: 'POST', body: fd });
                const result = await res.json();

                if (result.success) {
                    offset = result.next_offset;
                    imported += result.imported;
                    done = result.done;

                    const pct = Math.min(100, Math.round((offset / total) * 100));
                    document.getElementById('board-progress-bar').style.width = pct + '%';
                    document.getElementById('board-pct').textContent = pct + '%';
                    document.getElementById('board-counts').textContent = `${imported} / ${total}`;
                    document.getElementById('board-status').textContent = `Processing record ${offset}...`;

                    if (done) {
                        document.getElementById('board-progress-bar').style.background = '#48bb78';
                        const msg = document.getElementById('import-result-msg');
                        msg.className = 'alert alert-success';
                        msg.innerHTML = `<i class="fas fa-check-circle"></i> Bulk import successful! <strong>${imported}</strong> boards added.`;
                        msg.style.display = 'block';
                        setTimeout(() => location.reload(), 2000);
                    }
                } else {
                    throw new Error(result.error);
                }
            } catch (err) {
                document.getElementById('board-progress-bar').style.background = '#e53e3e';
                const msg = document.getElementById('import-result-msg');
                msg.className = 'alert alert-error';
                msg.innerHTML = `<i class="fas fa-exclamation-triangle"></i> <strong>Import Failed:</strong> ${err.message}`;
                msg.style.display = 'block';
                break;
            }
        }
    }
</script>

<?php require_once 'includes/footer.php'; ?>
