<?php
require_once __DIR__ . '/../config/db.php';
$pdo = getDBConnection();

// Fetch stages for calculation
$stages = $pdo->query("SELECT * FROM stages ORDER BY old_day DESC")->fetchAll();

// Filtering parameters
$reg_start = $_GET['reg_start'] ?? '';
$reg_end = $_GET['reg_end'] ?? '';
$last_start = $_GET['last_start'] ?? '';
$last_end = $_GET['last_end'] ?? '';
$filter_stage = $_GET['stage'] ?? '';

// Sorting parameters
$sort = $_GET['sort'] ?? 'last_pinned';
$order = $_GET['order'] ?? 'DESC';
$valid_sorts = ['acc_username', 'creation_date', 'last_pinned', 'total_posted', 'status'];
if (!in_array($sort, $valid_sorts)) $sort = 'last_pinned';
$order = ($order === 'ASC') ? 'ASC' : 'DESC';

// Build the base query
$where = [];
$params = [];

if ($reg_start) { $where[] = "MIN(ab.creation_date) >= :reg_start"; $params['reg_start'] = $reg_start; }
if ($reg_end) { $where[] = "MIN(ab.creation_date) <= :reg_end"; $params['reg_end'] = $reg_end; }
if ($last_start) { $where[] = "MAX(p.account_date) >= :last_start"; $params['last_start'] = $last_start; }
if ($last_end) { $where[] = "MAX(p.account_date) <= :last_end"; $params['last_end'] = $last_end; }

// Subquery approach for better filtering of aggregated results
$having = !empty($where) ? "HAVING " . implode(" AND ", $where) : "";

$query = "
    SELECT 
        ab.acc_username, 
        MIN(ab.creation_date) as creation_date, 
        MAX(p.account_date) as last_pinned,
        COUNT(p.id) as total_posted,
        DATEDIFF(CURRENT_DATE, MIN(ab.creation_date)) as account_age
    FROM account_board ab
    LEFT JOIN pins p ON ab.acc_username = p.acc_username
    GROUP BY ab.acc_username
    $having
";

// Wrap in a subquery to allow sorting by status and filtering by stage
$final_query = "SELECT * FROM ($query) as accounts_data";

// Sorting logic
$order_by = "ORDER BY $sort $order";
if ($sort === 'status') {
    // Custom status sort: Active Today first
    $order_by = "ORDER BY (last_pinned = CURRENT_DATE) $order, last_pinned $order";
}

$accounts = $pdo->prepare("$final_query $order_by");
$accounts->execute($params);
$results = $accounts->fetchAll();

// Post-fetch filtering for Posting Stage (calculated in PHP)
if ($filter_stage !== '') {
    $results = array_filter($results, function($acc) use ($stages, $filter_stage) {
        $current_stage = "Stage 0";
        foreach ($stages as $s) {
            if ($acc['account_age'] >= $s['old_day']) {
                $current_stage = $s['name'];
                break;
            }
        }
        return $current_stage === $filter_stage;
    });
}

function getSortUrl($field, $currentSort, $currentOrder) {
    $newOrder = ($field === $currentSort && $currentOrder === 'ASC') ? 'DESC' : 'ASC';
    $params = $_GET;
    $params['sort'] = $field;
    $params['order'] = $newOrder;
    return '?' . http_build_query($params);
}

function getSortIcon($field, $currentSort, $currentOrder) {
    if ($field !== $currentSort) return '<i class="fas fa-sort" style="opacity: 0.3; margin-left: 5px;"></i>';
    return $currentOrder === 'ASC' ? '<i class="fas fa-sort-up" style="margin-left: 5px;"></i>' : '<i class="fas fa-sort-down" style="margin-left: 5px;"></i>';
}

$page_title = 'Account Management - Pinterest Pin Manager';
require_once 'includes/header.php';
?>

<div class="header-content">
    <h1><i class="fas fa-user-circle"></i> Account Management</h1>
</div>

<!-- Filters Section -->
<div class="page-card">
    <h3 style="margin-bottom: 20px;"><i class="fas fa-filter"></i> Filters</h3>
    <form method="GET" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">
        <div class="form-group">
            <label>Registration From</label>
            <input type="date" name="reg_start" value="<?php echo htmlspecialchars($reg_start); ?>" class="form-control">
        </div>
        <div class="form-group">
            <label>Registration To</label>
            <input type="date" name="reg_end" value="<?php echo htmlspecialchars($reg_end); ?>" class="form-control">
        </div>
        <div class="form-group">
            <label>Last Pined From</label>
            <input type="date" name="last_start" value="<?php echo htmlspecialchars($last_start); ?>" class="form-control">
        </div>
        <div class="form-group">
            <label>Last Pined To</label>
            <input type="date" name="last_end" value="<?php echo htmlspecialchars($last_end); ?>" class="form-control">
        </div>
        <div class="form-group">
            <label>Posting Stage</label>
            <select name="stage" class="form-control">
                <option value="">All Stages</option>
                <?php 
                // Restore reverse order for stages dropdown
                $stages_asc = array_reverse($stages);
                foreach ($stages_asc as $s): ?>
                    <option value="<?php echo htmlspecialchars($s['name']); ?>" <?php echo $filter_stage === $s['name'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($s['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div style="display: flex; align-items: flex-end; gap: 10px;">
            <button type="submit" class="btn-submit" style="flex: 1;"><i class="fas fa-search"></i> Apply</button>
            <a href="accounts.php" class="btn-submit" style="background: #cbd5e0; color: #4a5568; flex: 1; text-decoration: none;"><i class="fas fa-undo"></i> Reset</a>
        </div>
    </form>
</div>

<div class="page-card">
    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
        <h2>Active Accounts (<?php echo count($results); ?>)</h2>
    </div>
    
    <div style="overflow-x: auto;">
        <table>
            <thead>
                <tr>
                    <th><a href="<?php echo getSortUrl('acc_username', $sort, $order); ?>" style="text-decoration: none; color: inherit;">Account Username <?php echo getSortIcon('acc_username', $sort, $order); ?></a></th>
                    <th><a href="<?php echo getSortUrl('creation_date', $sort, $order); ?>" style="text-decoration: none; color: inherit;">Registration Date <?php echo getSortIcon('creation_date', $sort, $order); ?></a></th>
                    <th><a href="<?php echo getSortUrl('last_pinned', $sort, $order); ?>" style="text-decoration: none; color: inherit;">Last Pinned <?php echo getSortIcon('last_pinned', $sort, $order); ?></a></th>
                    <th><a href="<?php echo getSortUrl('total_posted', $sort, $order); ?>" style="text-decoration: none; color: inherit;">Total Posted <?php echo getSortIcon('total_posted', $sort, $order); ?></a></th>
                    <th>Posting Stage</th>
                    <th><a href="<?php echo getSortUrl('status', $sort, $order); ?>" style="text-decoration: none; color: inherit;">Status <?php echo getSortIcon('status', $sort, $order); ?></a></th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($results)): ?>
                    <tr><td colspan="6" style="text-align: center; color: #999; padding: 40px;">No accounts found matching your filters.</td></tr>
                <?php else: ?>
                    <?php foreach ($results as $acc): 
                        $last_pinned = $acc['last_pinned'];
                        $is_active_today = ($last_pinned == date('Y-m-d'));
                        $status_class = $is_active_today ? 'badge-success' : 'badge-warning';
                        $status_text = $is_active_today ? 'Active Today' : 'Idle';
                        
                        // Calculate stage
                        $current_stage = "Stage 0";
                        foreach ($stages as $s) {
                            if ($acc['account_age'] >= $s['old_day']) {
                                $current_stage = $s['name'];
                                break;
                            }
                        }
                    ?>
                        <tr>
                            <td>
                                <div style="display: flex; align-items: center; gap: 10px;">
                                    <div style="width: 35px; height: 35px; background: #edf2f7; border-radius: 50%; display: flex; align-items: center; justify-content: center; color: var(--primary);">
                                        <i class="fas fa-user"></i>
                                    </div>
                                    <strong><?php echo htmlspecialchars($acc['acc_username']); ?></strong>
                                </div>
                            </td>
                            <td>
                                <div style="color: #4a5568; font-size: 0.9em;">
                                    <i class="far fa-calendar-alt" style="margin-right: 5px; opacity: 0.6;"></i>
                                    <?php echo date('M d, Y', strtotime($acc['creation_date'])); ?>
                                </div>
                            </td>
                            <td>
                                <?php if ($last_pinned): ?>
                                    <div style="color: #4a5568; font-size: 0.9em;">
                                        <i class="far fa-clock" style="margin-right: 5px; opacity: 0.6;"></i>
                                        <?php echo date('M d, Y', strtotime($last_pinned)); ?>
                                    </div>
                                <?php else: ?>
                                    <span style="color: #cbd5e0;">Never</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge badge-info"><?php echo $acc['total_posted']; ?> pins</span>
                            </td>
                            <td>
                                <span class="badge badge-info" style="background: #e2e8f0; color: #4a5568;"><?php echo htmlspecialchars($current_stage); ?></span>
                            </td>
                            <td>
                                <span class="badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php require_once 'includes/footer.php'; ?>
